/** @file   objective.h
 * @brief   Declaraction of Objective - class.
 * @version $Revision: 1.1.1.1 $
 * @author  Tomi Lamminsaari
 */

#ifndef H_WWW_OBJECTIVE_H
#define H_WWW_OBJECTIVE_H

#include <vector>
#include <string>
#include "rect2d.h"
#include "dynamicevent.h"
#include "gameobject.h"

namespace WeWantWar {



/** @class  Objective
 * @brief   Each of the level objectives are instances of this class.
 * @author  Tomi Lamminsaari
 *
 * A level can have several different objectives. To finish the level
 * player need to accomplish each of these objectives.
 *
 * Every objective is an instance of this class. The objectives can have
 * several finishing conditions such as player goes inside a certain area or
 * when timer has reached zero.
 *
 * All the possible finishing conditions are:
 * - Player goes inside a rectangular area.
 * - Timer has reached zero. There is also a TimerMode - member that tells if
 *   the objective is then failed or successful.
 * - There are no enemies inside a rectangular area.
 * - GameObject with certain ID-code has died.
 *
 * You can have several finishing conditions active at the same time.
 *
 * There's no need for setter - methods because the information about the
 * objectives are read from mapfiles.
 */
class Objective
{
public:

  ///
  /// Constants, datatype and static members
  /// ======================================

  /** Datatype for the Finishing Conditions. These are a bitwise flags. */
  typedef unsigned int FCond;

  /** Bitmask for condition that player gets inside a rectangle. */
  static const FCond FC_PLAYERINSIDE = 1;
  /** Bitmask for condition that timer has reached zero */
  static const FCond FC_TIMER = 1 << 1;
  /** Bitmask for condition that there are no enemies inside a rectangle. */
  static const FCond FC_NOENEMYINSIDE = 1 << 2;
  /** Bitmask for condition that all the objects with given IDCode are
   * dead
   */
  static const FCond FC_OBJECTSKILLED = 1 << 3;

  
  
  /** Datatype for the member defining the state we're operating in. */
  typedef int State;
  /** Objective has not been finished yet. */
  static const State INCOMPLETE = 0;
  /** Objective has been accomplished succefully. */
  static const State ACCOMPLISHED = 1;
  /** Objective has failed. */
  static const State FAILED = 2;
  
  
  /** Datatype for the mode we're operating */
  typedef int Mode;
  /** When a condition is realized, it means that the objective is complete. */
  static const Mode M_SUCCESSFUL = 0;
  /** When a condition is realized, it means that the objective failed. */
  static const Mode M_FAILS = 1;


  ///
  /// Constructors, destructor and operators
  /// ======================================

	/** Constructor.
   */
	Objective();


	/** Destructor
   */
	virtual ~Objective();

private:

	/** Copy constructor.
   * @param     rO                Reference to another Objective
   */
  Objective( const Objective& rO );

	/** Assignment operator
   * @param     rO                Reference to another Objective
   * @return    Reference to us.
   */
  Objective& operator = ( const Objective& rO );

public:


  ///
  /// Methods
  /// =======

  /** Updates this objective. Launches the Dynamic Events if needed.
   * @return    ACCOMPLISHED, FAILED or INCOMPLETE.
   */
  virtual State update();

  /** Reads the data of this objective from given stream. This method expects
   * that the opening OBJECTIVE - tag has been read away from the stream.
   * @param     rIn               The input stream
   * @return    Nonzero if fails.
   */
  virtual int readData( std::istream& rIn );
  
   


  ///
  /// Getter methods
  /// ==============

  /** Tells if this objective has been marked as accomplished.
   * @return    ACCOMPLISHED, FAILED or INCOMPLETE
   */
  virtual State state() const;
  
  
  /** Returns the timer.
   * @return    The timervalue. Nonzero if the timer-condition has not been
   *            activated yet.
   */
  virtual int getTimer() const;
  
  /** Returns the message
   * @return    The message that describes this objective.
   */
  virtual const std::string& getMessage() const;

  /** Returns the soundnumber that should be played when this objective has
   * been accomplished.
   * @return    The sound number or -1 if no sound should be played.
   */
  virtual int soundNumber() const;
  

protected:

  ///
  /// Protected interface
  /// ===================
  
  /** Checks if player is inside the @c m_playerRect - rectangle.
   * @return      <code>true</code> if player is inside it
   */
  bool playerInsideRect() const;
  
  
  /** Adds the given event to the Events-list. The ownership of the given
   * object is given to this Objective.
   * @param     pEvent            Pointer to the DynamicEvent being added.
   */
  void addEvent( DynamicEvent* pEvent );
  
  /** Reads the message.
   * @param     rIn               The input stream
   * @return    Nonzero if fails.
   */
  int readMessage( std::istream& rIn );
  
  /** Reads the datablock that contains the finishing conditions.
   * @param     rIn               The input stream
   * @return    Nonzero if fails.
   */
  int readFinishingConditions( std::istream& rIn );
  
  /** Reads the contents of COUNTDOWN - element.
   * @param     rIn               The input stream
   * @return    Nonzero if fails.
   */
  int readCountdown( std::istream& rIn );
  
  /** Reads the contents of PLAYER_INSIDE - element
   * @param     rIn               The input stream
   * @return    nonzero if fails.
   */
  int readPlayerInside( std::istream& rIn );
  
  /** Reads the contents of AREA_CLEAR - element.
   * @param     rIn               The input stream
   * @return    Nonzero if fails.
   */
  int readAreaClear( std::istream& rIn );
  
  /** Reads the contents of KILLED_ID - element
   * @param     rIn               The input stream
   * @return    Nonzero if fails.
   */
  int readKilledID( std::istream& rIn );
  
  
  ///
  /// Members
  /// =======


  /** Indicates if we are using the timer. */
  bool m_countdownFlag;
  
  /** The timer-value where the countdown starts. Decreased 40 times per
   * second.
   */
  int m_timer;

  /** When countdown reaches zero, this member tells if the objective is
   * then failed or successful.
   */
  Mode m_timermode;
  
  
  /** If this flag is set, then the objective can be accomplished by going
   * inside a rectangular area in the map.
   */
  bool m_playerInsideFlag;
  
  /** The player-rectangle */
  eng2d::Rect2D m_playerRect;
  
  
  /** If this flag is set, then the objective gets accomplished when there are
   * no enemies inside a rectangular area.
   */
  bool m_areaClearFlag;
  
  /** The No Enemies inside - rectangles */
  std::vector< eng2d::Rect2D > m_areaClearRects;
  
  /** ID - code of the GameObjects that are concerned. */
  GameObject::IDCode m_areaClearID;
  
  /* The area-clear mode. When there are no living objects inside the
   * rectangles, this member defines if the objective is complete or
   * if it failed.
   */
  Mode m_areaClearMode;
  

  
  /** If this flag is set, then when GameObject with certain ID-codes gets
   * killed, the objective is either failed or accomplished.
   */
  bool m_killedIDFlag;
  
  /** The id code of an object */
  std::vector<GameObject::IDCode> m_killedIDCodes;
  
  /** The objectid mode. When none of the objects with certain ID-codes are
   * alive, this member tell if the objective is accomplished or if it's
   * failed.
   */
  Mode m_killedIDMode;
  
  /** This timer tells the minimum duration the Objective can last. During
   * this time this objective ignores the finishing conditions.
   */
  int m_inactiveTimer;

protected:

  /** The Dynamic Events this objective has are stored in this vector. */
  std::vector< DynamicEvent* >  m_events;

  /** The state of this objective */
  State m_state;
  
  /** The message that describes this event. */
  std::string m_message;
  
  /** The sound number that should be played when this objective has been
   * accomplished.
   */
  int m_accomplishedSound;
  
  
  
private:

  ///
  /// Private members
  /// ===============

};

};  // end of namespace

#endif

/**
 * Version history
 * ===============
 * $Log: objective.h,v $
 * Revision 1.1.1.1  2006/01/21 23:02:42  lamminsa
 * no message
 *
 * Revision 1.0  2005-11-06 01:17:07+02  lamminsa
 * Initial revision
 *
 */
 
